/**
 * Advanced Simulation Module
 * @author Wen Xu
 *
 */
package pipe.modules.advancedsimulation;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingWorker;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import pipe.common.dataLayer.DataLayer;
import pipe.common.dataLayer.DataLayerInterface;
import pipe.common.dataLayer.DataLayerSimulationInterface;
import pipe.common.dataLayer.DataLayerWriter;
import pipe.common.dataLayer.Marking;
import pipe.common.dataLayer.Place;
import pipe.common.dataLayer.Transition;
import pipe.gui.CreateGui;
import pipe.gui.widgets.ButtonBar;
import pipe.gui.widgets.EscapableDialog;
import pipe.gui.widgets.PetriNetChooserPanel;
import pipe.gui.widgets.ResultsHTMLPane;
import pipe.modules.Module;

public class AdvancedSimulation extends SwingWorker implements Module {

	private static final String MODULE_NAME = "Advanced Simulation";

	private PetriNetChooserPanel sourceFilePanel;
	private ResultsHTMLPane results;

	public void run(DataLayer pnmlData) {
		EscapableDialog guiDialog = new EscapableDialog(CreateGui.getApp(),
				MODULE_NAME, true);

		// 1 Set layout
		Container contentPane = guiDialog.getContentPane();
		contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.PAGE_AXIS));

		// 2 Add file browser
		sourceFilePanel = new PetriNetChooserPanel("Source net", pnmlData);
		contentPane.add(sourceFilePanel);

		// 3 Add results pane
		results = new ResultsHTMLPane(pnmlData.getURI());
		contentPane.add(results);

		// 4 Add button
		contentPane.add(new ButtonBar("Simulate", simulateButtonClick,
				guiDialog.getRootPane()));

		// 5 Make window fit contents' preferred size
		guiDialog.pack();

		// 6 Move window to the middle of the screen
		guiDialog.setLocationRelativeTo(null);

		guiDialog.setVisible(true);
	}

	public String getName() {
		return MODULE_NAME;
	}

	// if (!sourceDataLayer.getPetriNetObjects().hasNext()) {

	/**
	 * Simulate button click handler
	 */
	ActionListener simulateButtonClick = new ActionListener() {

		public void actionPerformed(ActionEvent arg0) {
			DataLayerInterface sourceDataLayer = sourceFilePanel.getDataLayer();
			String s = "<h2>Petri net simulation results</h2>";
			if (sourceDataLayer == null) {
				JOptionPane.showMessageDialog(null,
						"Please, choose a source net", "Error",
						JOptionPane.ERROR_MESSAGE);
				return;
			}
			if (!sourceDataLayer.hasPlaceTransitionObjects()) {
				s += "No Petri net objects defined!";
			} else {
				try {

					DataLayerWriter.saveTemporaryFile(sourceDataLayer, this
							.getClass().getName());

					// simulate
					s += simulate(sourceDataLayer);
					results.setEnabled(true);
				} catch (NumberFormatException e) {
					s += "Invalid parameter!";
				} catch (OutOfMemoryError oome) {
					System.gc();
					results.setText("");
					s = "Memory error: " + oome.getMessage();

					s += "<br>Not enough memory. Please use a larger heap size."
							+ "<br>"
							+ "<br>Note:"
							+ "<br>The Java heap size can be specified with the -Xmx option."
							+ "<br>E.g., to use 512MB as heap size, the command line looks like this:"
							+ "<br>java -Xmx512m -classpath ...\n";
					results.setText(s);
					return;
				} catch (Exception e) {
					e.printStackTrace();
					s = "<br>Error" + e.getMessage();
					results.setText(s);
					return;
				}
			}
			results.setText(s);
		}
	};

	public String simulate(DataLayerInterface data) {

		String res = "";
		String transitionInfo = "";
		String simulateInfo = "";

		// transition info
		ArrayList resultStr = new ArrayList();

		resultStr.add("Name");
		resultStr.add("Execute Time");
		resultStr.add("Script Path");
		resultStr.add("Source Code Path");

		Transition[] trans = data.getTransitions();
		for (int i = 0; i < data.getTransitionsCount(); i++) {
			resultStr.add(trans[i].getName());
			resultStr.add(trans[i].getExecuteTime());
			resultStr.add(trans[i].getScriptPath());
			resultStr.add(trans[i].getSrcPath());
		}

		transitionInfo = ResultsHTMLPane.makeTable(resultStr.toArray(), 4,
				false, true, true, true);
		res += transitionInfo;

		// initialize
		ArrayList unfiredTrans = (ArrayList) data.getTransitionsArrayList()
				.clone();
		boolean keepRunning = true;
		resultStr = new ArrayList();

		// store initial marking
		data.storeState();
		
		// current time
		long start_time = System.currentTimeMillis();

		// simulate the net
		while (keepRunning) {

			ArrayList enabledTrans = data.getEnabledTransitions();

			if (enabledTrans.size() == 0 || unfiredTrans.size() == 0) {
				keepRunning = false;
				continue;
			}

			for (int i = 0; i < enabledTrans.size(); i++) {
				Transition curT = (Transition) enabledTrans.get(i);

				int exeTime = curT.getExecuteTime();
				long stop_time = System.currentTimeMillis();
				int elapsed = -1;
				// check time constraints
				if (curT.isTimed() && exeTime != -1 ) {
					elapsed = (int) ((stop_time - start_time) / 1000);
					if (elapsed < exeTime) {
						continue;
					}
				}
				
				// display info
				resultStr.add("\n");
				resultStr.add("Fired transition " + curT.getName() + " at time " + Integer.toString(elapsed));
				System.out.println("Fired transition " + curT.getName()  + " at time " + Integer.toString(elapsed));
				
				// running associated command on this transition
				// run script
				String scriptPath = curT.getScriptPath();
				// not empty
				if (scriptPath != null && !scriptPath.trim().equals("")) {
					resultStr.add(executeScript(scriptPath));
				}
				
				// run java source code
				String srcPath = curT.getSrcPath();
				// not empty
				if (srcPath != null && !srcPath.trim().equals("")) {
					resultStr.add(executeSrc(srcPath));
				}
	
				// fire current transition
				data.fireTransition(curT);
				// unfired
				if (unfiredTrans.contains(curT)) {
					unfiredTrans.remove(curT);
				}
			}
		}
		simulateInfo = ResultsHTMLPane.makeTable(resultStr.toArray(), 1, false,
				false, false, false);

		res += simulateInfo;

		// reset initial markings
		data.restoreState();

		return res;
	}

	@Override
	protected Object doInBackground() throws Exception {
		throw new UnsupportedOperationException("Not supported yet.");
	}
	
	private String executeScript(String scriptPath) {
		// slash
		String ostype = System.getProperty("os.name");
		String slash = "/";
		if (ostype.contains("Windows") || ostype.contains("windows") || ostype.contains("WINDOWS")) {
			slash= "\\";
		}
		
		String result = "Successfully executed " + scriptPath;
		try {
			// batch file
			if (scriptPath.endsWith(".bat")) {
				// temporary bat file
				String batfilename = "tmpsimulation.bat";
				File batfile = new File(batfilename);
				PrintWriter pwf = new PrintWriter(batfile);
				// access disk
				String disk = scriptPath.substring(0, 1);
				pwf.write(disk + ":\n");
				pwf.write("cd " + disk + ":\\" + "\n");
				// access directory
				String srcRemainPath = scriptPath.substring(0, scriptPath.lastIndexOf(slash) + 1);
				pwf.write("cd " + srcRemainPath + "\n");
				// run script
				pwf.write("cmd.exe /c start " + scriptPath + "\n");
				pwf.write("\n");
				pwf.close();
				
				StartProcessThread spThread = new StartProcessThread(batfilename) ;
				spThread.start();
			}
			else if (scriptPath.endsWith(".jar")) {
				// jar
				String cmd = "java -jar " + scriptPath;
				StartProcessThread spThread = new StartProcessThread(cmd) ;
				spThread.start();
			}
			else {
				// program
				StartProcessThread spThread = new StartProcessThread(scriptPath) ;
				spThread.start();
			}
		} catch (IOException e) {
			result = "Cannot execute " + scriptPath;
			e.printStackTrace();
		}
	
		return result;
	}
	
	private String executeSrc(String srcPath) {
		
		// slash
		String ostype = System.getProperty("os.name");
		String slash = "/";
		if (ostype.contains("Windows") || ostype.contains("windows") || ostype.contains("WINDOWS")) {
			slash= "\\";
		}
		
		String result = "Successfully executed " + srcPath;
		
		try {
			String className = srcPath.substring(srcPath.lastIndexOf(slash) + 1, srcPath.lastIndexOf("."));
			String srcRemainPath = srcPath.substring(0, srcPath.lastIndexOf(slash) + 1);
			// compile
			String cmd = "cmd.exe /k start javac " + "\"" + srcPath + "\"";
			Process pr_c = Runtime.getRuntime().exec(cmd);

			// run
			cmd = "cmd.exe /k start java -classpath " + srcRemainPath + " " + className;
			
			StartProcessThread spThread = new StartProcessThread(cmd) ;
			spThread.start();

		} catch (IOException e) {
			result = "Cannot execute " + srcPath;
			e.printStackTrace();
		}
	
		return result;
	}
}

class StartProcessThread extends Thread{
	
	private String cmd;
	private Process process;
	
	public StartProcessThread(String cmd) {
		this.cmd = cmd;
	}

	public void run() {
		try {
			process = Runtime.getRuntime().exec(cmd);
			while (true) {
				try {
					process.exitValue();
					break;
				} catch (IllegalThreadStateException e) {
					InputStream stream = process.getInputStream();
					if (stream.available() > 0) {
						byte[] btmp = new byte[stream.available()];
						stream.read(btmp);
					}
				}
				Thread.sleep(500);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}	
}
