///////////////////////////////////////////////////////////////////////////////////////
// Test GestureRecognizer - SceneDrawer from OpenNI demo
///////////////////////////////////////////////////////////////////////////////////////

#include <XnOpenNI.h>
#include <XnCodecIDs.h>

#include <string>
#include <signal.h>
#include <ctime>

#include <GL/glut.h>

#include "SceneDrawer.h"
#include "GestureRecognizer.h"

XnBool g_bDrawBackground = TRUE;
XnBool g_bDrawPixels = TRUE;
XnBool g_bDrawSkeleton = TRUE;
XnBool g_bPrintID = TRUE;
XnBool g_bPrintState = TRUE;

#define GL_WIN_SIZE_X 720
#define GL_WIN_SIZE_Y 480

XnChar g_strPose[20] = "";
XnBool g_bPause = false;
XnBool g_bRecord = false;

XnBool g_bQuit = false;

GestureRecognizer _recognizer;

void CleanupExit(int i = 0)
{
	_recognizer.shutdown();
	exit (1);
}

// this function is called each frame
void glutDisplay (void)
{

	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Setup the OpenGL viewpoint
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();

	xn::SceneMetaData sceneMD;
	xn::DepthMetaData depthMD;
	_recognizer.depthGenerator().GetMetaData(depthMD);
	glOrtho(0, depthMD.XRes(), depthMD.YRes(), 0, -1.0, 1.0);

	glDisable(GL_TEXTURE_2D);

	if (!g_bPause)
	{
		// Read next available data
		_recognizer.waitAndUpdateAll();
	}

    // Process the data
    _recognizer.depthGenerator().GetMetaData(depthMD);
    _recognizer.userGenerator().GetUserPixels(0, sceneMD);
    SceneDrawer::DrawDepthMap(depthMD, sceneMD);

	glutSwapBuffers();
}

void glutIdle (void)
{
	if (g_bQuit) {
		CleanupExit();
	}

	// Display the frame
	glutPostRedisplay();
}

void glutKeyboard (unsigned char key, int x, int y)
{
    static int deg = 0;
	switch (key)
	{
	case 27:
		CleanupExit();
	case 'b':
		// Draw background?
		g_bDrawBackground = !g_bDrawBackground;
		break;
	case 'x':
		// Draw pixels at all?
		g_bDrawPixels = !g_bDrawPixels;
		break;
	case 's':
		// Draw Skeleton?
		g_bDrawSkeleton = !g_bDrawSkeleton;
		break;
	case 'i':
		// Print label?
		g_bPrintID = !g_bPrintID;
		break;
	case 'l':
		// Print ID & state as label, or only ID?
		g_bPrintState = !g_bPrintState;
		break;
	case'p':
		g_bPause = !g_bPause;
		break;
	}
}

void glInit (int * pargc, char ** argv)
{
	glutInit(pargc, argv);
	glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
	glutInitWindowSize(GL_WIN_SIZE_X, GL_WIN_SIZE_Y);
	glutCreateWindow ("User Tracker Derivative");
	//glutFullScreen();
	glutSetCursor(GLUT_CURSOR_NONE);

	glutKeyboardFunc(glutKeyboard);
	glutDisplayFunc(glutDisplay);
	glutIdleFunc(glutIdle);

	glDisable(GL_DEPTH_TEST);
	glEnable(GL_TEXTURE_2D);

	glEnableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
}


// Callback: New user was detected
void XN_CALLBACK_TYPE User_NewUser(xn::UserGenerator& generator, XnUserID nId, void* pCookie)
{
	printf("New User %d\n", nId);
	// New user found
	if (_recognizer.needPose())
	{
		_recognizer.getPoseDetectionCapability().StartPoseDetection(_recognizer.getPoseName().c_str(), nId);
	}
	else
	{
		_recognizer.getSkeletonCapability().RequestCalibration(nId, TRUE);
	}
}

// Callback: An existing user was lost
void XN_CALLBACK_TYPE User_LostUser(xn::UserGenerator& generator, XnUserID nId, void* pCookie)
{
	printf("Lost user %d\n", nId);
}

// Callback: Detected a pose
void XN_CALLBACK_TYPE UserPose_PoseDetected(xn::PoseDetectionCapability& capability, const XnChar* strPose, XnUserID nId, void* pCookie)
{
	printf("Pose %s detected for user %d\n", strPose, nId);
	_recognizer.getPoseDetectionCapability().StopPoseDetection(nId);
	_recognizer.getSkeletonCapability().RequestCalibration(nId, TRUE);
}

// Callback: Started calibration
void XN_CALLBACK_TYPE UserCalibration_CalibrationStart(xn::SkeletonCapability& capability, XnUserID nId, void* pCookie)
{
	printf("Calibration started for user %d\n", nId);
    //memset(SceneDrawer::strVector, 0, sizeof(SceneDrawer::strVector));
    //sprintf(SceneDrawer::strVector, "Calibration started for user %d", nId);
}

// Callback: Finished calibration
void XN_CALLBACK_TYPE UserCalibration_CalibrationEnd(xn::SkeletonCapability& capability, XnUserID nId, XnBool bSuccess, void* pCookie)
{
	if (bSuccess)
	{
		// Calibration succeeded
		printf("Calibration complete, start tracking user %d\n", nId);
		_recognizer.getSkeletonCapability().StartTracking(nId);
	}
	else
	{
		// Calibration failed
		printf("Calibration failed for user %d\n", nId);
		if (_recognizer.needPose())
		{
			_recognizer.getPoseDetectionCapability().StartPoseDetection(_recognizer.getPoseName().c_str(), nId);
		}
		else
		{
			_recognizer.getSkeletonCapability().RequestCalibration(nId, TRUE);
		}
	}
}

// Callback: Created Hand
void XN_CALLBACK_TYPE Hands_HandCreate(xn::HandsGenerator& generator, XnUserID nId, const XnPoint3D *pPosition, XnFloat fTime, void* pCookie)
{
    printf("[hands] Hand create %d\n", nId);
}

void XN_CALLBACK_TYPE Hands_HandDestroy(xn::HandsGenerator& generator, XnUserID nId, XnFloat fTime, void* pCookie)
{
    printf("[hands] Hand Destroy %d\n", nId);
    //set label on scene for instructions
    char buf[100];
    sprintf(buf, "Wave to begin hand tracking OR make position to begin body tracking.");
    SceneDrawer::SetLabel("instr", buf, 20, 20);
    _recognizer.addGesture("Wave", NULL);
    _recognizer.addGesture("Click", NULL);
    _recognizer.addGesture("Circle", NULL);
}

void newpoint(const XnPoint3D *point, const XnVector3D& vel, const XnVector3D& accel)
{
    static XnPoint3D p;
    if(point->Y < p.Y) printf("[newpoint] down (%f) ", vel.Y);
    else { printf("[newpoint] up (%f) ", vel.Y); }

    if(point->X < p.X) printf("[newpoint] left (%f) ", vel.X);
    else printf("[newpoint] right (%f) ", vel.X);

    if(point->Z < p.Z) printf("[newpoint] in (%f) ", vel.Z);
    else printf("[newpoint] back (%f) ", vel.Z);
    
    printf("\n");
    p = *point;
}

// This is just a hack example of some stuff you might do
// with the position/time data.
XnVector3D _prev_vel;
XnPoint3D _prev_loc;
XnFloat _prev_time;
void XN_CALLBACK_TYPE Hands_HandUpdate(xn::HandsGenerator& generator, XnUserID nId, const XnPoint3D *pPosition, XnFloat fTime, void* pCookie)
{
    printf("Hand update [%lf]: <%f, %f, %f> / [%lf]: <%f, %f, %f>\n", fTime, pPosition->X, pPosition->Y, pPosition->Z,
        _prev_time, _prev_loc.X, _prev_loc.Y, _prev_loc.Z);
    XnVector3D delta_pos = *pPosition;
    delta_pos.X -= _prev_loc.X;
    delta_pos.Y -= _prev_loc.Y;
    delta_pos.Z -= _prev_loc.Z;
    // movement threshold
    static const XnFloat MOVE_THRESH = 2.5;
    if(delta_pos.X * delta_pos.X < MOVE_THRESH) delta_pos.X = 0;
    if(delta_pos.Y * delta_pos.Y < MOVE_THRESH) delta_pos.Y = 0;
    if(delta_pos.Z * delta_pos.Z < MOVE_THRESH) delta_pos.Z = 0;
    XnFloat delta_time = fTime - _prev_time;
    XnVector3D vel = delta_pos;
    if(vel.X < 0) vel.X *= -1;
    if(vel.Y < 0) vel.Y *= -1;
    if(vel.Z < 0) vel.Z *= -1;
    vel.X /= delta_time;
    vel.Y /= delta_time;
    vel.Z /= delta_time;
    XnVector3D accel;
    accel.X = (vel.X - _prev_vel.X)/delta_time;
    accel.Y = (vel.Y - _prev_vel.Y)/delta_time;
    accel.Z = (vel.Z - _prev_vel.Z)/delta_time;
        
    // print the position vector
    //memset(SceneDrawer::strAccel, 0, sizeof(SceneDrawer::strVector));
    //sprintf(SceneDrawer::strAccel, "position. <%f, %f, %f>", pPosition->X, pPosition->Y, pPosition->Z);
    
    
    //Example on how to create a label for the SceneDrawer
    char str[100];
    sprintf(str, "position. <%f, %f, %f>", pPosition->X, pPosition->Y, pPosition->Z);
    SceneDrawer::SetLabel("accel", str, 20, 40); //label name, text, xposition, y position

    static const XnFloat DELTA_THRESH = 9;
    printf("<delta x> %f\n", delta_pos.X);
    if(delta_pos.X > DELTA_THRESH ||
       delta_pos.Y > DELTA_THRESH ||
       delta_pos.Z > DELTA_THRESH)
    {
        newpoint(pPosition, vel, accel);
    }

    /*
    static const XnFloat TIME_THRESH = .25,
                         POS_THRESH  = .125;

    if(delta_time > TIME_THRESH && // time threshold
       delta_pos.X*delta_pos.X > POS_THRESH) // position threshold
    {
        newpoint(pPosition, vel, accel);
        _prev_time = fTime;
        memset(SceneDrawer::strVector, 0, sizeof(SceneDrawer::strVector));
        sprintf(SceneDrawer::strVector, "vel. <%f, %f, %f>", vel.X, vel.Y, vel.Z);
        memset(SceneDrawer::strAccel, 0, sizeof(SceneDrawer::strVector));
        sprintf(SceneDrawer::strAccel, "accel. <%f, %f, %f>", accel.X, accel.Y, accel.Z);
    }
    */
    _prev_time = fTime;
    _prev_loc = *pPosition;
    _prev_vel = vel;
}

void XN_CALLBACK_TYPE Gesture_Recognized(xn::GestureGenerator& generator, const XnChar* strGesture, const XnPoint3D* pIDPosition, const XnPoint3D* pEndPosition, void* pCookie)
{
    static int i = 0;
    printf("Gesture Recognized: %s\n", strGesture);

    // create label with the gesture name and time on the scene
    time_t t;
    time(&t);
    tm *local = localtime(&t);
    char tbuf[50];
    strftime(tbuf, sizeof(tbuf), "%I:%M.%S%p", local);
    
    char lbl[100];    
    sprintf(lbl, "Gesture: %s recognized @ %s", strGesture, tbuf);
    SceneDrawer::SetLabel("gesture", lbl, 20, 60);

    // Here, the wave initiates hand tracking
    if(!strcmp(strGesture, "Wave"))
    {
        SceneDrawer::SetLabel("instr", "");
        printf("Beginning hand tracking..\n");
        //_recognizer.removeGesture(strGesture);
        _recognizer.startTrackingHands(*pEndPosition);
    }
    ++i;
}

void XN_CALLBACK_TYPE Gesture_Progress(xn::GestureGenerator& generator, const XnChar* strGesture, const XnPoint3D* pPosition, const XnFloat progress, void* pCookie)
{
    printf("Gesture Progress\n");
}

#define SAMPLE_XML_PATH "Config.xml"
#define CHECK_RC(nRetVal, what)										\
	if (nRetVal != XN_STATUS_OK)									\
	{																\
		printf("%s failed: %s\n", what, xnGetStatusString(nRetVal));\
		return nRetVal;												\
	}

int main(int argc, char **argv)
{
    bool invisible = true;

    signal(SIGINT, CleanupExit);

	XnStatus nRetVal = XN_STATUS_OK;

	if (argc > 1)
	{
        if('i' == *argv[1])
        {
            invisible = true;
            nRetVal = _recognizer.init(SAMPLE_XML_PATH);
            CHECK_RC(nRetVal, "InitFromXml");
        }
	}
	else
	{
        nRetVal = _recognizer.init(SAMPLE_XML_PATH);
		CHECK_RC(nRetVal, "InitFromXml");
	}

    XnCallbackHandle hSkeletonCallbacks, hPoseCallbacks;
	_recognizer.registerUserCallbacks(User_NewUser, User_LostUser);
	_recognizer.getSkeletonCapability().RegisterCalibrationCallbacks(UserCalibration_CalibrationStart, UserCalibration_CalibrationEnd, NULL, hSkeletonCallbacks);

    _recognizer.getPoseDetectionCapability().RegisterToPoseCallbacks(UserPose_PoseDetected, NULL, NULL, hPoseCallbacks);
    _recognizer.setSkeletonProfile(XN_SKEL_PROFILE_ALL);
    _recognizer.registerHandCallbacks(Hands_HandCreate, Hands_HandUpdate, Hands_HandDestroy);
    _recognizer.registerGestureCallbacks(Gesture_Recognized, Gesture_Progress);

	nRetVal = _recognizer.startGeneratingAll();
	CHECK_RC(nRetVal, "StartGenerating");

    // add gestures
    _recognizer.addGesture("Wave", NULL);
    _recognizer.addGesture("Click", NULL);
    _recognizer.addGesture("Circle", NULL);
    //create  label for scene directions
    char buf[100];
    sprintf(buf, "Wave to begin hand tracking OR make position to begin body tracking.");
    SceneDrawer::SetLabel("instr", buf, 20, 20);

    glInit(&argc, argv);
    glutMainLoop();
}
