#include <jni.h>
#include <cstdio>
#include <iostream>

#include <cstdarg> // debug_printf

#include <XnCppWrapper.h>
#include <XnOpenNI.h>
#include <XnCodecIDs.h>

#include "GestureRecognizer.h"
#include "HandTracker.h"

#define XML_PATH "Config.xml"

GestureRecognizer _grec;

static JavaVM *_jvm; // safe to cache (same across threads)
// cached method ids - these are ok to cache
static jmethodID j_GestureRecognizedCB,
                 j_HandUpdateCB,
                 j_Point3DCtor;

// Point3D field ids
static jfieldID j_PointX, j_PointY, j_PointZ;      

// weak global refs:
static jobject j_HandCallbackObj,
               j_GestureCallbackObj;


enum PrintType
{
    PrintErr,
    PrintDebug
};

int debug_printf(PrintType p, const char *fmt, ...)
{
    FILE *fp = stdout;
    const char *type = "";
    switch(p)
    {
        case PrintErr:
            fp = stderr;
            type = "[HandTracker - native] ";
            break;
    }

    va_list lst;
    va_start(lst, fmt);
    int ret = vfprintf(fp, fmt, lst);
    va_end(lst);
    return ret;
}

JNIEnv *_get_env_for_current_thread(JavaVM *const jvm)
{
    if(!jvm) return NULL;
    JNIEnv *env;
    jvm->AttachCurrentThread((void **)&env, NULL);
    return env;
}

JNIEnv *j_env()
{
    return _get_env_for_current_thread(_jvm);
}

void XN_CALLBACK_TYPE Hands_HandUpdate(xn::HandsGenerator& generator, XnUserID nId, const XnPoint3D *pPosition, XnFloat fTime, void* pCookie)
{
    debug_printf(PrintDebug, "hand update\n");
    if(!j_HandCallbackObj) return;

    JNIEnv *env = j_env();
    jclass jPoint3D = env->FindClass("Point3D");
    // check callback object reference is live
    if( !jPoint3D ||
        !j_HandUpdateCB)
    {
        debug_printf(PrintErr, "hand update - class or callback null");
        return;
    }

    jobject jpos = env->NewObject(jPoint3D, j_Point3DCtor, pPosition->X, pPosition->Y, pPosition->Z);
    env->CallVoidMethod(j_HandCallbackObj, j_HandUpdateCB, nId, jpos, fTime);

    env->DeleteLocalRef(jpos);
    env->DeleteLocalRef(jPoint3D);
}

void XN_CALLBACK_TYPE Hands_HandCreate(xn::HandsGenerator& generator, XnUserID nId, const XnPoint3D *pPosition, XnFloat fTime, void* pCookie)
{
    debug_printf(PrintDebug, "[hands] Hand create %d\n", nId);
}

void XN_CALLBACK_TYPE Hands_HandDestroy(xn::HandsGenerator& generator, XnUserID nId, XnFloat fTime, void* pCookie)
{
    printf("[hands] Hand Destroy %d\n", nId);
    _grec.addGesture("Wave", NULL);
    _grec.addGesture("Click", NULL);
}

void XN_CALLBACK_TYPE Gesture_Recognized(xn::GestureGenerator& generator, const XnChar* strGesture, const XnPoint3D* pIDPosition, const XnPoint3D* pEndPosition, void* pCookie)
{
    printf("[HandTracker - native] Gesture Recognized: %s\n", strGesture);
    if(!j_GestureCallbackObj) return;

    JNIEnv *env = j_env();
    jclass jPoint3D = env->FindClass("Point3D");
    if(!jPoint3D ||
        !j_GestureRecognizedCB)
    {
        debug_printf(PrintErr, "Gesture_Recognized - could not find class or method for gesture callback\n");
        return;
    }

    jobject jpoint = env->NewObject(jPoint3D, j_Point3DCtor, (jfloat)pIDPosition->X, (jfloat)pIDPosition->Y, (jfloat)pIDPosition->Z);
    if(!jpoint)
    {
        debug_printf(PrintErr, "error constructing point for callback\n");
        return;
    }
    env->CallVoidMethod(j_GestureCallbackObj, j_GestureRecognizedCB, env->NewStringUTF(strGesture), jpoint);

    env->DeleteLocalRef(jPoint3D);
    env->DeleteLocalRef(jpoint);

    // Here, the wave initiates hand tracking
    if(!strcmp(strGesture, "Wave"))
    {
        debug_printf(PrintDebug, "Beginning hand tracking..\n");
        _grec.startTrackingHands(*pEndPosition);
    }
}

void XN_CALLBACK_TYPE Gesture_Progress(xn::GestureGenerator& generator, const XnChar* strGesture, const XnPoint3D* pPosition, const XnFloat progress, void* pCookie)
{
    printf("Gesture Progress\n");
}

void Java_HandTracker_waitOnce(JNIEnv *env, jobject obj)
{
    _grec.waitAndUpdateAll();
}

void Java_HandTracker_RegisterGestureCallbackObject(JNIEnv *env, jobject obj, jobject cb)
{
    jclass jHandTrackerCls = env->FindClass("IGestureCallbackObject");
    if(!jHandTrackerCls) return; // exception thrown
    j_GestureCallbackObj = env->NewGlobalRef(cb); // cb is pinned until the object is unregistered!
    if(!j_GestureCallbackObj)
    {
        debug_printf(PrintErr, "could not get gesture recognized cb\n");
        return;
    }
    // cache method IDs, too
    j_GestureRecognizedCB = env->GetMethodID(jHandTrackerCls, "GestureRecognized", "(Ljava/lang/String;LPoint3D;)V");

    env->DeleteLocalRef(jHandTrackerCls);
}

void Java_HandTracker_UnRegisterGestureCallbackObject(JNIEnv *env, jobject obj)
{
    if(j_GestureCallbackObj)
    {
        env->DeleteGlobalRef(j_GestureCallbackObj);
        j_GestureCallbackObj = NULL;
        debug_printf(PrintDebug, "freed gesture callback object\n");
    }
}

void Java_HandTracker_RegisterHandCallbackObject(JNIEnv *env, jobject obj, jobject cb)
{
    jclass jHandTrackerCls = env->FindClass("IHandCallbackObject");
    if(!jHandTrackerCls) return; // exception thrown
    j_HandCallbackObj = env->NewGlobalRef(cb); // pinned
    if(!j_HandCallbackObj)
    {
        debug_printf(PrintErr, "[HandTracker - native] could not get hand cb object\n");
        return;
    }
    // cache method IDs, too
    j_HandUpdateCB = env->GetMethodID(jHandTrackerCls, "HandUpdate", "(ILPoint3D;F)V");

    env->DeleteLocalRef(jHandTrackerCls);
}

void Java_HandTracker_UnRegisterHandCallbackObject(JNIEnv *env, jobject obj)
{
    if(j_HandCallbackObj)
    {
        env->DeleteGlobalRef(j_HandCallbackObj);
        j_HandCallbackObj = NULL;
        debug_printf(PrintDebug, "freed hand callback object\n");
    }
}

void Java_HandTracker_init(JNIEnv *env, jobject obj)
{
    // try to get the jvm instance so we can
    // get the current thread's enviroment from
    // the callbacks
    jint ret = env->GetJavaVM(&_jvm);
    if(0 != ret) return;

    // if other threads reset these, they
    // will still have the same value
    jobject jHandTracker = obj;

    jclass jPoint3D = env->FindClass("Point3D");
    if(!jPoint3D)
    {
        debug_printf(PrintErr, "Could not get Point3D constructor id\n");
        return;
    }

    // cache point constructor
    j_Point3DCtor = env->GetMethodID(jPoint3D, "<init>", "(FFF)V");
    if(!j_Point3DCtor)
    {
        debug_printf(PrintErr, "Could not get Point3D constructor id\n");
        return;
    }

    // cache point fields
    j_PointX = env->GetFieldID(jPoint3D, "X", "F");
    j_PointY = env->GetFieldID(jPoint3D, "Y", "F");
    j_PointZ = env->GetFieldID(jPoint3D, "Z", "F");
    if(!j_PointX || !j_PointY || !j_PointZ)
    {
        debug_printf(PrintErr, "Could not get Point3D fields\n");
        return;
    }

    // we wait to start up the recognizer until we
    // know we can get all of the JNI stuff setup/cached
    XnStatus nRetVal = _grec.init(XML_PATH);
    if(nRetVal != XN_STATUS_OK)
    {
		debug_printf(PrintErr, "%s Initialization failed\n", xnGetStatusString(nRetVal));
        return;
    }

    _grec.registerHandCallbacks(Hands_HandCreate, Hands_HandUpdate, Hands_HandDestroy);
    _grec.registerGestureCallbacks(Gesture_Recognized, Gesture_Progress);

    _grec.addGesture("Wave");
    _grec.addGesture("Click");

	nRetVal = _grec.startGeneratingAll();
}

void Java_HandTracker_shutDown(JNIEnv *env, jclass obj)
{
    Java_HandTracker_UnRegisterHandCallbackObject(env, obj);
    Java_HandTracker_UnRegisterGestureCallbackObject(env, obj);
    _grec.shutdown();
}

